import os
import asyncio
from PyQt5 import QtWidgets
from PyQt5.QtWidgets import QFileDialog, QMessageBox, QCheckBox, QTextEdit
from PyQt5.QtCore import Qt
import librosa
from mutagen.easyid3 import EasyID3
from mutagen.mp3 import MP3
from mutagen.flac import FLAC
from mutagen.wave import WAVE
from shazamio import Shazam


class MP3MetadataApp(QtWidgets.QWidget):
    def __init__(self):
        super().__init__()
        self.init_ui()

    def init_ui(self):
        self.layout = QtWidgets.QVBoxLayout()
        self.dir_layout = QtWidgets.QHBoxLayout()
        self.buttons_layout = QtWidgets.QHBoxLayout()

        self.label = QtWidgets.QLabel("Select Directory:")
        self.dir_line_edit = QtWidgets.QLineEdit()
        self.browse_button = QtWidgets.QPushButton("Browse")
        self.process_button = QtWidgets.QPushButton("Process Audio Files")
        self.progress_bar = QtWidgets.QProgressBar()
        self.include_bpm_checkbox = QCheckBox("Include BPM as prefix in filename")
        self.include_bpm_checkbox.setChecked(True)
        self.log_text_area = QTextEdit()
        self.log_text_area.setReadOnly(True)

        self.dir_layout.addWidget(self.dir_line_edit)
        self.dir_layout.addWidget(self.browse_button)
        self.layout.addWidget(self.label)
        self.layout.addLayout(self.dir_layout)
        self.layout.addWidget(self.include_bpm_checkbox)
        self.layout.addWidget(self.process_button)
        self.layout.addWidget(self.progress_bar)
        self.layout.addWidget(self.log_text_area)

        self.setLayout(self.layout)
        self.setWindowTitle("Audio Metadata Processor")
        self.setGeometry(300, 300, 500, 400)

        self.progress_bar.setValue(0)
        self.progress_bar.setAlignment(Qt.AlignCenter)

        self.browse_button.clicked.connect(self.browse_directory)
        self.process_button.clicked.connect(self.process_files)

    def log_message(self, message):
        self.log_text_area.append(message)
        self.log_text_area.ensureCursorVisible()

    def browse_directory(self):
        dir_path = QFileDialog.getExistingDirectory(self, "Select Directory")
        if dir_path:
            self.dir_line_edit.setText(dir_path)

    def process_files(self):
        directory = self.dir_line_edit.text().strip()
        if not directory:
            QMessageBox.warning(self, "Error", "Please select a directory first.")
            return

        if not os.path.exists(directory):
            QMessageBox.warning(self, "Error", "The selected directory does not exist.")
            return

        include_bpm = self.include_bpm_checkbox.isChecked()
        asyncio.run(self.process_audio_files(directory, include_bpm))
        QMessageBox.information(self, "Success", "Files have been processed and metadata updated!")

    async def process_audio_files(self, directory, include_bpm):
        shazam = Shazam()
        files = [f for f in os.listdir(directory) if f.lower().endswith(('.mp3', '.flac', '.wav'))]
        total_files = len(files)

        if total_files == 0:
            QMessageBox.warning(self, "Error", "No audio files found in the selected directory.")
            return

        self.progress_bar.setMaximum(total_files)
        self.progress_bar.setValue(0)

        for i, file_name in enumerate(files):
            file_path = os.path.join(directory, file_name)
            self.log_message(f"Processing: {file_name}")

            try:
                # Shazam recognition
                with open(file_path, "rb") as audio_file:
                    audio_bytes = audio_file.read()
                result = await shazam.recognize(audio_bytes)

                title, artist, album = "Unknown Title", "Unknown Artist", "Unknown Album"
                if "track" in result:
                    track = result["track"]
                    title = track.get("title", "Unknown Title")
                    artist = track.get("subtitle", "Unknown Artist")
                    album = track.get("sections", [{}])[0].get("metadata", [{}])[0].get("text", "Unknown Album")

                # Calculate BPM
                bpm = self.calculate_bpm(file_path)
                if bpm is not None:
                    bpm = int(bpm)

                # Skip processing if any metadata is unknown
                if title == "Unknown Title" or artist == "Unknown Artist" or album == "Unknown Album" or bpm is None:
                    self.log_message(f"Skipping file '{file_name}' due to missing metadata.")
                    self.progress_bar.setValue(i + 1)
                    continue

                # Update metadata
                self.update_metadata(file_path, title, artist, album, bpm)

                # Rename file if include_bpm is True
                if include_bpm:
                    new_file_name = f"{bpm}_{artist} - {title}{os.path.splitext(file_name)[1]}"
                else:
                    new_file_name = f"{artist} - {title}{os.path.splitext(file_name)[1]}"

                # Clean filename
                new_file_name = "".join(c for c in new_file_name if c.isalnum() or c in " -_.")
                new_file_path = os.path.join(directory, new_file_name)
                os.rename(file_path, new_file_path)

                self.log_message(f"Renamed to: {new_file_name}")
                self.log_message("####################################")
            except Exception as e:
                self.log_message(f"Error processing {file_name}: {e}")

            self.progress_bar.setValue(i + 1)

    def calculate_bpm(self, file_path):
        try:
            # Load audio file using librosa
            y, sr = librosa.load(file_path, sr=None)
            onset_env = librosa.onset.onset_strength(y=y, sr=sr)
            tempo = librosa.beat.tempo(onset_envelope=onset_env, sr=sr, aggregate=None)
            return tempo[0] if len(tempo) > 0 else None
        except Exception as e:
            self.log_message(f"Error calculating BPM for {file_path}: {e}")
            return None

    def update_metadata(self, file_path, title, artist, album, bpm):
        try:
            ext = os.path.splitext(file_path)[1].lower()
            if ext == ".mp3":
                audio = MP3(file_path, ID3=EasyID3)
            elif ext == ".flac":
                audio = FLAC(file_path)
            elif ext == ".wav":
                audio = WAVE(file_path)
            else:
                self.log_message(f"Unsupported file format: {file_path}")
                return

            audio["title"] = title
            audio["artist"] = artist
            audio["album"] = album
            if bpm is not None:
                audio["bpm"] = str(bpm)
            audio.save()
            self.log_message(f"Updated metadata.")
        except Exception as e:
            self.log_message(f"Error updating metadata for '{file_path}': {e}")


def main():
    app = QtWidgets.QApplication([])
    window = MP3MetadataApp()
    window.show()
    app.exec_()


if __name__ == "__main__":
    main()
