#!/usr/bin/env python3
# -*- coding: utf-8 -*-

"""
Smart Image Compressor
---------------------
A GUI application for compressing images with format-specific handling:
- JPG files can be converted to WebP for better compression (optional)
- PNG files are compressed but kept as PNG to preserve transparency

Features:
- Select single/multiple files or a folder (recursive)
- Toggle WebP conversion on/off
- Smart handling of different image formats
- Progress tracking and detailed log

Dependencies:
- PyQt5
- Pillow (PIL)
"""

import sys
import os
import gc
from PIL import Image
from PyQt5.QtWidgets import (QApplication, QMainWindow, QPushButton, QFileDialog,
                             QLabel, QProgressBar, QVBoxLayout, QHBoxLayout,
                             QWidget, QListWidget, QCheckBox, QSpinBox, QMessageBox,
                             QRadioButton, QButtonGroup, QGroupBox, QTextEdit)
from PyQt5.QtCore import Qt, QThread, pyqtSignal


class ImageProcessorThread(QThread):
    """Worker thread to process images without blocking the UI"""
    progress_update = pyqtSignal(int)
    file_processed = pyqtSignal(str, bool, int, int, str)  # filename, success, orig_size, new_size, message
    finished_signal = pyqtSignal(int, int, int, int)  # success_count, failed_count, total_orig_size, total_new_size

    def __init__(self, file_list, output_dir, compression_mode, quality, smart_compression, use_webp):
        super().__init__()
        self.file_list = file_list
        self.output_dir = output_dir  # Not used for in-place compression
        self.is_lossless = (compression_mode == "lossless")
        self.quality = quality
        self.smart_compression = smart_compression
        self.use_webp = use_webp
        self.success_count = 0
        self.failed_count = 0
        self.total_original_size = 0
        self.total_new_size = 0

    def run(self):
        """Process all files in the queue"""
        total_files = len(self.file_list)

        for i, file_path in enumerate(self.file_list):
            # Update progress
            progress = int((i / total_files) * 100) if total_files > 0 else 0
            self.progress_update.emit(progress)

            # Process the file
            self._process_file(file_path)

            # Clean up memory
            gc.collect()

        # Final progress update
        self.progress_update.emit(100)

        # Emit finished signal with statistics
        self.finished_signal.emit(
            self.success_count,
            self.failed_count,
            self.total_original_size,
            self.total_new_size
        )

    def _process_file(self, file_path):
        """Process a single file with proper error handling"""
        # Log which file is being processed to identify problematic images
        print(f"Starting to process: {file_path}")

        # File validation
        if not os.path.exists(file_path) or not os.path.isfile(file_path):
            self.file_processed.emit(
                os.path.basename(file_path),
                False,
                0,
                0,
                "File not found or is not a valid file"
            )
            self.failed_count += 1
            return

        # Get original size and file extension
        try:
            original_size = os.path.getsize(file_path)
            print(f"File size: {self._format_size(original_size)}")
            self.total_original_size += original_size
            file_ext = os.path.splitext(file_path)[1].lower()
        except OSError as e:
            self.file_processed.emit(
                os.path.basename(file_path),
                False,
                0,
                0,
                f"Error reading file: {str(e)}"
            )
            self.failed_count += 1
            return

        # Determine if we need to create a temporary file (for in-place compression)
        try:
            # Determine file format based on extension
            is_jpg = file_ext in ['.jpg', '.jpeg', '.jfif']
            is_png = file_ext in ['.png']

            # Check if we need to change the format (only for JPG to WebP)
            format_change = is_jpg and self.use_webp

            if format_change:
                # If changing format, create new path with new extension
                file_name_without_ext = os.path.splitext(file_path)[0]
                output_ext = '.webp'
                output_path = f"{file_name_without_ext}{output_ext}"
            else:
                # For in-place compression, create a temporary file first
                output_path = f"{file_path}.temp"
        except Exception as e:
            self.file_processed.emit(
                os.path.basename(file_path),
                False,
                original_size,
                0,
                f"Error creating output path: {str(e)}"
            )
            self.failed_count += 1
            return

        # Process the image
        img = None
        result = None

        try:
            # Open the image with logging for DecompressionBombWarning
            import warnings
            warnings.filterwarnings('error', category=Image.DecompressionBombWarning)

            try:
                print(f"Attempting to open image: {file_path}")
                img = Image.open(file_path)

                # Log image dimensions to identify large images
                width, height = img.size
                pixel_count = width * height
                print(f"Image dimensions: {width}x{height} = {pixel_count} pixels")

                # Check if this is the problematic image
                if pixel_count > 89_478_485:  # PIL's default limit
                    print(f"WARNING: FOUND LARGE IMAGE exceeding PIL limit: {file_path}")
                    print(f"This is the image causing DecompressionBombWarning!")

            except Warning as w:
                # Catch the DecompressionBombWarning
                print(f"WARNING CAUGHT for {file_path}: {str(w)}")
                print("This is the image causing the DecompressionBombWarning!")

                # Re-open the image, suppressing the warning
                warnings.filterwarnings('ignore', category=Image.DecompressionBombWarning)
                img = Image.open(file_path)

                # Log image dimensions
                width, height = img.size
                pixel_count = width * height
                print(f"Large image dimensions: {width}x{height} = {pixel_count} pixels")

            # Reset warning filter to default
            warnings.resetwarnings()

            # Continue with normal processing
            if is_jpg and self.use_webp:
                # Process JPGs - convert to WebP if enabled
                result = self._process_jpg_to_webp(img, file_path, output_path, original_size)
            elif is_jpg and not self.use_webp:
                # Process JPGs but keep as JPG
                result = self._process_jpg_to_jpg(img, file_path, output_path, original_size)
            elif is_png:
                # Process PNGs - keep as PNG but compress
                result = self._process_png(img, file_path, output_path, original_size)
            else:
                # For other formats, just try basic optimization
                result = self._process_other_format(img, file_path, output_path, original_size)

            # Process result
            if result['success']:
                # If we created a temp file and it was successful, replace the original
                if not format_change and os.path.exists(output_path):
                    # Make sure to close the image before replacing
                    if img:
                        img.close()
                        img = None

                    # Replace original with optimized version
                    try:
                        os.replace(output_path, file_path)
                    except Exception as e:
                        # If replace fails, try remove and rename
                        os.remove(file_path)
                        os.rename(output_path, file_path)

                # Update stats
                self.success_count += 1
                self.total_new_size += result['new_size']

                # Calculate compression percentage
                compression_percent = round((1 - (result['new_size'] / original_size)) * 100, 2)

                # Emit signal with result
                self.file_processed.emit(
                    os.path.basename(file_path),
                    True,
                    original_size,
                    result['new_size'],
                    f"{result['action']}: {self._format_size(original_size)} → {self._format_size(result['new_size'])} ({compression_percent}% smaller)"
                )
            else:
                # Clean up any temporary files
                if os.path.exists(output_path) and output_path != file_path:
                    try:
                        os.remove(output_path)
                    except:
                        pass

                self.failed_count += 1
                self.file_processed.emit(
                    os.path.basename(file_path),
                    False,
                    original_size,
                    0,
                    result['message']
                )

        except Exception as e:
            print(f"ERROR processing {file_path}: {str(e)}")

            # Clean up any temporary files
            if os.path.exists(output_path) and output_path != file_path:
                try:
                    os.remove(output_path)
                except:
                    pass

            self.failed_count += 1
            self.file_processed.emit(
                os.path.basename(file_path),
                False,
                original_size,
                0,
                f"Error processing image: {str(e)}"
            )

        finally:
            # Clean up resources
            if img:
                img.close()
                del img

            print(f"Finished processing: {file_path}")
            print("-" * 50)

    def _process_jpg_to_jpg(self, img, file_path, output_path, original_size):
        """Optimize JPG without converting to WebP"""
        try:
            # Save with optimization
            img.save(output_path, format="JPEG", optimize=True, quality=self.quality)
            new_size = os.path.getsize(output_path)

            if new_size < original_size:
                return {
                    'success': True,
                    'new_size': new_size,
                    'action': "JPG optimized"
                }
            else:
                # If optimization didn't help, don't replace the original
                if os.path.exists(output_path):
                    os.remove(output_path)

                return {
                    'success': True,
                    'new_size': original_size,
                    'action': "JPG preserved (no size reduction possible)"
                }
        except Exception as e:
            if os.path.exists(output_path):
                try:
                    os.remove(output_path)
                except:
                    pass

            return {
                'success': False,
                'message': f"Error optimizing JPG: {str(e)}"
            }

    def _process_jpg_to_webp(self, img, file_path, output_path, original_size):
        """Convert JPG to WebP with compression options"""
        if self.smart_compression:
            return self._apply_smart_webp_compression(img, file_path, output_path, original_size)
        else:
            return self._apply_simple_webp_compression(img, file_path, output_path, original_size)

    def _process_png(self, img, file_path, output_path, original_size):
        """Process PNG images - optimize but keep as PNG"""
        try:
            # Create a temporary copy for optimization
            if os.path.exists(output_path):
                os.remove(output_path)

            # Try to optimize the PNG
            img.save(output_path, format="PNG", optimize=True)
            new_size = os.path.getsize(output_path)

            # If optimized version is not smaller, try with reduced colors for non-transparent PNGs
            if new_size >= original_size and img.mode != 'RGBA' and 'transparency' not in img.info:
                try:
                    # Try with palette mode (reduces colors)
                    palette_img = img.convert('P', palette=Image.ADAPTIVE, colors=256)
                    palette_img.save(output_path, format="PNG", optimize=True)
                    palette_size = os.path.getsize(output_path)

                    if palette_size < new_size:
                        new_size = palette_size
                except:
                    pass

            # If no optimization method worked, don't replace the original
            if new_size >= original_size:
                # Remove the temporary file
                if os.path.exists(output_path):
                    os.remove(output_path)

                return {
                    'success': True,
                    'new_size': original_size,
                    'action': "PNG preserved (no size reduction possible)"
                }
            else:
                return {
                    'success': True,
                    'new_size': new_size,
                    'action': "PNG optimized"
                }
        except Exception as e:
            if os.path.exists(output_path):
                try:
                    os.remove(output_path)
                except:
                    pass

            return {
                'success': False,
                'message': f"Error optimizing PNG: {str(e)}"
            }

    def _process_other_format(self, img, file_path, output_path, original_size):
        """Process other image formats with basic optimization"""
        try:
            # Get format from original file
            format_name = img.format

            # Try to save with optimization
            img.save(output_path, format=format_name, optimize=True)
            new_size = os.path.getsize(output_path)

            if new_size < original_size:
                return {
                    'success': True,
                    'new_size': new_size,
                    'action': f"{format_name} optimized"
                }
            else:
                # If optimization didn't help, don't replace the original
                if os.path.exists(output_path):
                    os.remove(output_path)

                return {
                    'success': True,
                    'new_size': original_size,
                    'action': f"{format_name} preserved (no size reduction possible)"
                }
        except Exception as e:
            if os.path.exists(output_path):
                try:
                    os.remove(output_path)
                except:
                    pass

            return {
                'success': False,
                'message': f"Error processing {img.format}: {str(e)}"
            }

    def _apply_smart_webp_compression(self, img, file_path, output_path, original_size):
        """Apply different WebP compression methods and choose the best one"""
        best_size = float('inf')
        best_method = None
        best_quality = None
        methods_tried = []

        # Method 1: Original requested method
        try:
            img.save(output_path, format="WEBP", lossless=self.is_lossless, quality=self.quality)
            size1 = os.path.getsize(output_path)
            methods_tried.append(("original", size1))

            if size1 < best_size:
                best_size = size1
                best_method = "original"
                best_quality = "lossless" if self.is_lossless else self.quality
        except Exception:
            pass

        # Method 2: If lossless requested, try high quality lossy
        if self.is_lossless:
            try:
                img.save(output_path, format="WEBP", lossless=False, quality=95)
                size2 = os.path.getsize(output_path)
                methods_tried.append(("high_quality", size2))

                if size2 < best_size:
                    best_size = size2
                    best_method = "high_quality"
                    best_quality = 95
            except Exception:
                pass

        # Method 3: Medium quality
        try:
            img.save(output_path, format="WEBP", lossless=False, quality=75)
            size3 = os.path.getsize(output_path)
            methods_tried.append(("medium_quality", size3))

            if size3 < best_size:
                best_size = size3
                best_method = "medium_quality"
                best_quality = 75
        except Exception:
            pass

        # Method 4: Lower quality (only if others didn't give good results)
        if best_size >= original_size:
            try:
                img.save(output_path, format="WEBP", lossless=False, quality=60)
                size4 = os.path.getsize(output_path)
                methods_tried.append(("low_quality", size4))

                if size4 < best_size:
                    best_size = size4
                    best_method = "low_quality"
                    best_quality = 60
            except Exception:
                pass

        # Process result
        if best_size < original_size:
            # Re-save with best method if not already the last one tried
            if methods_tried and methods_tried[-1][0] != best_method:
                try:
                    if best_method == "original" and self.is_lossless:
                        img.save(output_path, format="WEBP", lossless=True)
                    else:
                        img.save(output_path, format="WEBP", lossless=False, quality=best_quality)
                except Exception as e:
                    return {
                        'success': False,
                        'message': f"Error saving with best method: {str(e)}"
                    }

            return {
                'success': True,
                'new_size': best_size,
                'action': "JPG converted to WebP"
            }
        else:
            # If file exists, remove it
            if os.path.exists(output_path):
                try:
                    os.remove(output_path)
                except:
                    pass

            # Keep original JPG since WebP wasn't smaller
            return {
                'success': True,
                'new_size': original_size,
                'action': "JPG preserved (WebP would be larger)"
            }

    def _apply_simple_webp_compression(self, img, file_path, output_path, original_size):
        """Apply a single WebP compression method"""
        try:
            img.save(output_path, format="WEBP", lossless=self.is_lossless, quality=self.quality)
            new_size = os.path.getsize(output_path)

            if new_size < original_size:
                return {
                    'success': True,
                    'new_size': new_size,
                    'action': "JPG converted to WebP"
                }
            else:
                # Remove the WebP file if it's larger
                if os.path.exists(output_path):
                    try:
                        os.remove(output_path)
                    except:
                        pass

                # Keep original JPG since WebP would be larger
                return {
                    'success': True,
                    'new_size': original_size,
                    'action': "JPG preserved (WebP would be larger)"
                }
        except Exception as e:
            return {
                'success': False,
                'message': f"Error converting to WebP: {str(e)}"
            }

    def _format_size(self, size_bytes):
        """Format file size in human-readable format"""
        if size_bytes < 1024:
            return f"{size_bytes} B"
        elif size_bytes < 1024 * 1024:
            return f"{size_bytes / 1024:.1f} KB"
        else:
            return f"{size_bytes / (1024 * 1024):.2f} MB"


class ImageCompressorApp(QMainWindow):
    """Main application window"""

    def __init__(self):
        super().__init__()
        self.setWindowTitle("Smart Image Compressor")
        self.setMinimumSize(800, 600)

        # Initialize variables
        self.file_list = []
        self.worker_thread = None

        # Set up the UI
        self.init_ui()

    def init_ui(self):
        """Initialize the user interface"""
        # Main layout
        main_widget = QWidget()
        main_layout = QVBoxLayout(main_widget)

        # File selection section
        file_selection_group = QGroupBox("File Selection")
        file_selection_layout = QVBoxLayout()

        # Selection mode
        selection_mode_layout = QHBoxLayout()
        self.single_file_radio = QRadioButton("Single/Multiple Files")
        self.folder_radio = QRadioButton("Folder (recursive scan)")
        self.single_file_radio.setChecked(True)

        selection_mode_group = QButtonGroup()
        selection_mode_group.addButton(self.single_file_radio)
        selection_mode_group.addButton(self.folder_radio)

        selection_mode_layout.addWidget(self.single_file_radio)
        selection_mode_layout.addWidget(self.folder_radio)
        file_selection_layout.addLayout(selection_mode_layout)

        # Browse buttons
        browse_layout = QHBoxLayout()
        self.browse_button = QPushButton("Browse Files")
        self.browse_folder_button = QPushButton("Browse Folder")

        browse_layout.addWidget(self.browse_button)
        browse_layout.addWidget(self.browse_folder_button)
        file_selection_layout.addLayout(browse_layout)

        # In-place compression note
        in_place_note = QLabel("Note: Files will be compressed in-place (originals will be replaced)")
        in_place_note.setStyleSheet("color: #d00; font-weight: bold;")
        file_selection_layout.addWidget(in_place_note)

        file_selection_group.setLayout(file_selection_layout)
        main_layout.addWidget(file_selection_group)

        # Compression options
        compression_group = QGroupBox("Compression Options")
        compression_layout = QVBoxLayout()

        # WebP conversion checkbox
        self.webp_checkbox = QCheckBox("Convert JPG files to WebP format")
        self.webp_checkbox.setChecked(False)
        self.webp_checkbox.setToolTip("When enabled, JPG files will be converted to WebP for better compression")
        compression_layout.addWidget(self.webp_checkbox)

        # Format info label
        format_info_label = QLabel(
            "• PNG files will be optimized but kept as PNG to preserve transparency\n"
            "• Other formats will be optimized with their original format"
        )
        format_info_label.setStyleSheet("color: #555; font-style: italic;")
        compression_layout.addWidget(format_info_label)

        # Compression type
        compression_type_layout = QHBoxLayout()
        self.lossless_radio = QRadioButton("Lossless")
        self.lossy_radio = QRadioButton("Lossy with Quality")
        self.lossy_radio.setChecked(True)  # Default to lossy

        compression_type_group = QButtonGroup()
        compression_type_group.addButton(self.lossless_radio)
        compression_type_group.addButton(self.lossy_radio)

        compression_type_layout.addWidget(self.lossless_radio)
        compression_type_layout.addWidget(self.lossy_radio)
        compression_layout.addLayout(compression_type_layout)

        # Quality settings
        quality_layout = QHBoxLayout()
        quality_layout.addWidget(QLabel("Quality (1-100):"))
        self.quality_spinner = QSpinBox()
        self.quality_spinner.setRange(1, 100)
        self.quality_spinner.setValue(85)  # Default quality
        quality_layout.addWidget(self.quality_spinner)
        compression_layout.addLayout(quality_layout)

        # Smart compression option
        self.smart_compression_checkbox = QCheckBox("Use smart compression (try multiple methods)")
        self.smart_compression_checkbox.setChecked(True)
        self.smart_compression_checkbox.setToolTip(
            "Automatically try different compression methods to find the smallest file size"
        )
        compression_layout.addWidget(self.smart_compression_checkbox)

        compression_group.setLayout(compression_layout)
        main_layout.addWidget(compression_group)

        # File list
        file_list_group = QGroupBox("Selected Files")
        file_list_layout = QVBoxLayout()

        self.file_list_widget = QListWidget()
        self.file_list_widget.setSelectionMode(QListWidget.ExtendedSelection)
        file_list_layout.addWidget(self.file_list_widget)

        file_actions_layout = QHBoxLayout()
        self.file_count_label = QLabel("0 files selected")
        self.remove_selected_button = QPushButton("Remove Selected")
        self.remove_selected_button.setEnabled(False)
        self.clear_all_button = QPushButton("Clear All")
        self.clear_all_button.setEnabled(False)

        file_actions_layout.addWidget(self.file_count_label)
        file_actions_layout.addWidget(self.remove_selected_button)
        file_actions_layout.addWidget(self.clear_all_button)
        file_list_layout.addLayout(file_actions_layout)

        file_list_group.setLayout(file_list_layout)
        main_layout.addWidget(file_list_group)

        # Progress section
        progress_group = QGroupBox("Progress")
        progress_layout = QVBoxLayout()

        self.progress_bar = QProgressBar()
        self.progress_bar.setValue(0)
        progress_layout.addWidget(self.progress_bar)

        # Log output
        self.log_text = QTextEdit()
        self.log_text.setReadOnly(True)
        self.log_text.setLineWrapMode(QTextEdit.WidgetWidth)
        self.log_text.setStyleSheet("font-family: monospace;")
        self.log_text.setText("Ready. Add files and select output directory to begin.")
        progress_layout.addWidget(self.log_text)

        progress_group.setLayout(progress_layout)
        main_layout.addWidget(progress_group)

        # Action buttons
        action_layout = QHBoxLayout()
        self.convert_button = QPushButton("Compress Images")
        self.convert_button.setEnabled(False)
        self.convert_button.setStyleSheet("font-weight: bold; padding: 8px;")

        action_layout.addStretch()
        action_layout.addWidget(self.convert_button)
        action_layout.addStretch()
        main_layout.addLayout(action_layout)

        # Set central widget
        self.setCentralWidget(main_widget)

        # Connect signals
        self.browse_button.clicked.connect(self.browse_files)
        self.browse_folder_button.clicked.connect(self.browse_folder)
        self.convert_button.clicked.connect(self.start_conversion)
        self.remove_selected_button.clicked.connect(self.remove_selected_files)
        self.clear_all_button.clicked.connect(self.clear_all_files)
        self.file_list_widget.itemSelectionChanged.connect(self.update_remove_button_state)

        # Connect WebP checkbox to update format info label
        self.webp_checkbox.toggled.connect(self.update_format_info)

    def update_format_info(self, checked):
        """Update format info label based on WebP conversion setting"""
        format_info_text = ""
        if checked:
            format_info_text = (
                "• JPG files will be converted to WebP for better compression\n"
                "• PNG files will be optimized but kept as PNG to preserve transparency\n"
                "• Other formats will be optimized with their original format"
            )
        else:
            format_info_text = (
                "• JPG files will be optimized but kept as JPG\n"
                "• PNG files will be optimized but kept as PNG to preserve transparency\n"
                "• Other formats will be optimized with their original format"
            )

        # Find the format info label and update it
        for i in range(self.centralWidget().layout().count()):
            widget = self.centralWidget().layout().itemAt(i).widget()
            if isinstance(widget, QGroupBox) and widget.title() == "Compression Options":
                for j in range(widget.layout().count()):
                    item = widget.layout().itemAt(j)
                    if item.widget() and isinstance(item.widget(), QLabel) and "PNG files" in item.widget().text():
                        item.widget().setText(format_info_text)
                        break
                break

    def closeEvent(self, event):
        """Handle window close event"""
        if self.worker_thread and self.worker_thread.isRunning():
            # Ask user if they want to cancel the conversion
            reply = QMessageBox.question(
                self,
                "Conversion in Progress",
                "Image compression is still running. Do you want to cancel it and exit?",
                QMessageBox.Yes | QMessageBox.No,
                QMessageBox.No
            )

            if reply == QMessageBox.Yes:
                self.worker_thread.terminate()
                self.worker_thread.wait()
                event.accept()
            else:
                event.ignore()
        else:
            event.accept()

    def browse_files(self):
        """Open file dialog to select image files"""
        file_paths, _ = QFileDialog.getOpenFileNames(
            self,
            "Select Images",
            "",
            "Image Files (*.png *.jpg *.jpeg *.gif *.bmp *.tiff *.webp *.ico);;All Files (*)"
        )

        if file_paths:
            self.single_file_radio.setChecked(True)
            self.add_files_to_list(file_paths)

    def browse_folder(self):
        """Open folder dialog to select a folder of images"""
        folder_path = QFileDialog.getExistingDirectory(self, "Select Folder")

        if folder_path:
            self.folder_radio.setChecked(True)

            # Get all image files from the selected folder and its subfolders
            image_extensions = ['.png', '.jpg', '.jpeg', '.gif', '.bmp', '.tiff', '.webp', '.ico']
            file_paths = []

            try:
                # Walk through all subfolders
                for root, dirs, files in os.walk(folder_path):
                    for file in files:
                        if any(file.lower().endswith(ext) for ext in image_extensions):
                            file_path = os.path.join(root, file)
                            file_paths.append(file_path)

                if file_paths:
                    self.add_files_to_list(file_paths)
                    self.log_message(f"Found {len(file_paths)} images in folder and subfolders")
                else:
                    QMessageBox.information(
                        self,
                        "No Images Found",
                        "No image files were found in the selected folder or its subfolders."
                    )
            except Exception as e:
                QMessageBox.critical(
                    self,
                    "Error",
                    f"Failed to read folder: {str(e)}"
                )

    def add_files_to_list(self, file_paths):
        """Add files to the file list"""
        # Add new files that are not already in the list
        added_count = 0
        for file_path in file_paths:
            if file_path not in self.file_list:
                self.file_list.append(file_path)
                self.file_list_widget.addItem(file_path)
                added_count += 1

        # Update UI
        self.update_file_count_label()
        self.update_convert_button_state()
        self.clear_all_button.setEnabled(len(self.file_list) > 0)

        # Log
        if added_count > 0:
            self.log_message(f"Added {added_count} files to the queue")

    def update_file_count_label(self):
        """Update the file count label"""
        count = len(self.file_list)
        self.file_count_label.setText(f"{count} file{'s' if count != 1 else ''} selected")

    def update_remove_button_state(self):
        """Update the state of the remove button"""
        self.remove_selected_button.setEnabled(len(self.file_list_widget.selectedItems()) > 0)

    def update_convert_button_state(self):
        """Update the state of the convert button"""
        self.convert_button.setEnabled(len(self.file_list) > 0)

    def remove_selected_files(self):
        """Remove selected files from the list"""
        selected_items = self.file_list_widget.selectedItems()

        if not selected_items:
            return

        for item in selected_items:
            row = self.file_list_widget.row(item)
            self.file_list_widget.takeItem(row)
            self.file_list.remove(item.text())

        self.update_file_count_label()
        self.update_convert_button_state()
        self.clear_all_button.setEnabled(len(self.file_list) > 0)

        self.log_message(f"Removed {len(selected_items)} file(s) from the queue")

    def clear_all_files(self):
        """Clear all files from the list"""
        if not self.file_list:
            return

        count = len(self.file_list)
        self.file_list_widget.clear()
        self.file_list = []

        self.update_file_count_label()
        self.update_convert_button_state()
        self.clear_all_button.setEnabled(False)

        self.log_message(f"Cleared all {count} file(s) from the queue")

    def start_conversion(self):
        """Start the conversion process"""
        # Validate inputs
        if not self.file_list:
            QMessageBox.warning(self, "No Files", "Please select at least one image file.")
            return

        # Show warning about in-place compression
        reply = QMessageBox.warning(
            self,
            "Confirm In-Place Compression",
            "This will compress files in-place, replacing the originals. This cannot be undone. Continue?",
            QMessageBox.Yes | QMessageBox.No,
            QMessageBox.No
        )

        if reply == QMessageBox.No:
            return

        # Disable UI during conversion
        self.set_ui_enabled(False)

        # Get compression options
        compression_mode = "lossless" if self.lossless_radio.isChecked() else "lossy"
        quality = self.quality_spinner.value()
        smart_compression = self.smart_compression_checkbox.isChecked()
        use_webp = self.webp_checkbox.isChecked()

        # Clear log and add start message
        self.log_text.clear()
        self.log_message(f"Starting compression of {len(self.file_list)} files...")

        # Log compression settings
        if use_webp:
            self.log_message("- JPG files will be converted to WebP for better compression")
        else:
            self.log_message("- JPG files will be optimized but kept as JPG")

        self.log_message("- PNG files will be optimized but kept as PNG to preserve transparency")

        if smart_compression:
            self.log_message("Smart compression is enabled - will try multiple methods for optimal results")

        # Create worker thread
        self.worker_thread = ImageProcessorThread(
            self.file_list.copy(),
            "",  # No output directory for in-place compression
            compression_mode,
            quality,
            smart_compression,
            use_webp
        )

        # Connect signals
        self.worker_thread.progress_update.connect(self.update_progress)
        self.worker_thread.file_processed.connect(self.file_processed)
        self.worker_thread.finished_signal.connect(self.conversion_finished)
        self.worker_thread.finished.connect(self.thread_cleanup)

        # Start thread
        self.worker_thread.start()

    def update_progress(self, value):
        """Update the progress bar"""
        self.progress_bar.setValue(value)

    def log_message(self, message):
        """Add a message to the log"""
        self.log_text.append(message)

        # Scroll to bottom
        scrollbar = self.log_text.verticalScrollBar()
        scrollbar.setValue(scrollbar.maximum())

    def file_processed(self, filename, success, orig_size, new_size, message):
        """Handle processed file update"""
        if success:
            self.log_message(f"✅ {filename}: {message}")
        else:
            self.log_message(f"⚠️ {filename}: {message}")

    def thread_cleanup(self):
        """Clean up thread resources"""
        if self.worker_thread:
            self.worker_thread.deleteLater()
            self.worker_thread = None

    def conversion_finished(self, success_count, failed_count, total_orig_size, total_new_size):
        """Handle conversion completion"""
        # Re-enable UI
        self.set_ui_enabled(True)

        # Force garbage collection
        gc.collect()

        # Log results
        self.log_message("")
        self.log_message("=" * 40)
        self.log_message(f"Compression completed: {success_count} succeeded, {failed_count} skipped/failed")

        if success_count > 0 and total_orig_size > 0 and total_new_size > 0:
            reduction = total_orig_size - total_new_size
            percent = round((reduction / total_orig_size) * 100, 2)

            orig_size_fmt = self.format_size(total_orig_size)
            new_size_fmt = self.format_size(total_new_size)
            saved_fmt = self.format_size(reduction)

            self.log_message(f"Total size reduction: {orig_size_fmt} → {new_size_fmt}")
            self.log_message(f"Space saved: {saved_fmt} ({percent}%)")

        # Show appropriate message dialog
        if success_count > 0 and failed_count == 0:
            QMessageBox.information(
                self,
                "Compression Complete",
                f"{success_count} images were compressed successfully."
            )
        elif success_count > 0 and failed_count > 0:
            QMessageBox.warning(
                self,
                "Compression Completed with Issues",
                f"{success_count} images were compressed successfully, but {failed_count} were skipped or failed."
            )
        elif success_count == 0:
            QMessageBox.critical(
                self,
                "Compression Failed",
                "No images were compressed successfully. Please check the log for details."
            )

    def set_ui_enabled(self, enabled):
        """Enable or disable UI elements during conversion"""
        self.browse_button.setEnabled(enabled)
        self.browse_folder_button.setEnabled(enabled)
        self.lossless_radio.setEnabled(enabled)
        self.lossy_radio.setEnabled(enabled)
        self.quality_spinner.setEnabled(enabled)
        self.smart_compression_checkbox.setEnabled(enabled)
        self.webp_checkbox.setEnabled(enabled)  # Enable/disable WebP checkbox
        self.file_list_widget.setEnabled(enabled)
        self.remove_selected_button.setEnabled(enabled and len(self.file_list_widget.selectedItems()) > 0)
        self.clear_all_button.setEnabled(enabled and len(self.file_list) > 0)
        self.convert_button.setEnabled(enabled and len(self.file_list) > 0)

        if not enabled:
            self.progress_bar.setValue(0)

    def format_size(self, size_bytes):
        """Format file size in human-readable format"""
        if size_bytes < 1024:
            return f"{size_bytes} B"
        elif size_bytes < 1024 * 1024:
            return f"{size_bytes / 1024:.1f} KB"
        else:
            return f"{size_bytes / (1024 * 1024):.2f} MB"


if __name__ == "__main__":
    app = QApplication(sys.argv)
    window = ImageCompressorApp()
    window.show()
    sys.exit(app.exec_())