import sys
import os
import cv2
import shutil
import numpy as np
import rawpy
from PIL import Image
from PyQt5.QtWidgets import (
    QApplication, QWidget, QPushButton, QFileDialog, QLabel, QVBoxLayout,
    QLineEdit, QTextEdit, QHBoxLayout, QProgressBar, QComboBox, QCheckBox
)
from PyQt5.QtGui import QPixmap, QImage
from PyQt5.QtCore import Qt, QThread, pyqtSignal

# Supported Image Formats
SUPPORTED_FORMATS = ('.png', '.jpg', '.jpeg', '.bmp', '.tiff', '.webp', '.cr2', '.nef', '.arw', '.dng', '.orf', '.raf')


def pil_to_qpixmap(pil_image):
    if pil_image.mode != "RGB":
        pil_image = pil_image.convert("RGB")

    img_data = pil_image.tobytes("raw", "RGB")
    qimage = QImage(img_data, pil_image.width, pil_image.height, pil_image.width * 3, QImage.Format_RGB888)

    return QPixmap.fromImage(qimage)


class ImageAnalyzerThread(QThread):
    progress = pyqtSignal(int)
    finished = pyqtSignal(list)

    def __init__(self, folder, threshold, methods, scan_subfolders):
        super().__init__()
        self.folder = folder
        self.threshold = threshold
        self.methods = methods
        self.scan_subfolders = scan_subfolders

    def run(self):
        blurry_images = []
        files = []

        if self.scan_subfolders:
            # Walk through all subdirectories
            for root, _, filenames in os.walk(self.folder):
                if "Blurry_Images" in root:
                    continue

                for filename in filenames:
                    if filename.lower().endswith(SUPPORTED_FORMATS):
                        files.append(os.path.join(root, filename))
        else:
            # Only scan the selected directory
            for filename in os.listdir(self.folder):
                if filename.lower().endswith(SUPPORTED_FORMATS):
                    files.append(os.path.join(self.folder, filename))

        for i, file_path in enumerate(files):
            image = BlurryImageDetector.load_image(file_path)

            if image is not None and self.is_blurry(image):
                blurry_images.append(file_path)

            self.progress.emit(int((i + 1) / len(files) * 100))

        self.finished.emit(blurry_images)

    def is_blurry(self, image):
        results = []

        if 'laplacian' in self.methods:
            laplacian_var = cv2.Laplacian(image, cv2.CV_64F).var()
            results.append(laplacian_var < self.threshold)

        if 'sobel' in self.methods:
            sobel_x = cv2.Sobel(image, cv2.CV_64F, 1, 0, ksize=5)
            sobel_y = cv2.Sobel(image, cv2.CV_64F, 0, 1, ksize=5)

            edge_var_x = np.var(sobel_x)
            edge_var_y = np.var(sobel_y)
            motion_blur_score = abs(edge_var_x - edge_var_y)

            edge_energy = np.mean(np.abs(sobel_x) + np.abs(sobel_y))
            noise_energy = np.var(image)
            enr = edge_energy / (noise_energy + 1e-6)  # Avoid division by zero

            is_motion_blurry = motion_blur_score > (self.threshold * 2) and enr < 0.1
            results.append(is_motion_blurry)

        return any(results)


class BlurryImageDetector(QWidget):
    def __init__(self):
        super().__init__()
        self.init_ui()
        self.selected_folder = None
        self.blurry_images = []
        self.current_index = -1
        self.analyzer_thread = None

    def init_ui(self):
        self.setWindowTitle("Advanced Blurry Image Detector")
        self.setGeometry(300, 200, 800, 700)

        layout = QVBoxLayout()

        folder_layout = QHBoxLayout()
        self.folder_label = QLabel("Selected Folder: None")
        self.browse_button = QPushButton("Browse Folder")
        self.browse_button.clicked.connect(self.browse_folder)
        folder_layout.addWidget(self.folder_label)
        folder_layout.addWidget(self.browse_button)
        layout.addLayout(folder_layout)

        # Add subfolder scanning option
        self.scan_subfolders_check = QCheckBox("Scan Subfolders")
        self.scan_subfolders_check.setChecked(True)  # Default to scanning subfolders
        layout.addWidget(self.scan_subfolders_check)

        methods_layout = QHBoxLayout()
        self.laplacian_check = QCheckBox("Laplacian (Defocus Blur)")
        self.laplacian_check.setChecked(True)
        self.sobel_check = QCheckBox("Sobel (Motion Blur)")
        self.sobel_check.setChecked(True)
        methods_layout.addWidget(self.laplacian_check)
        methods_layout.addWidget(self.sobel_check)
        layout.addLayout(methods_layout)

        threshold_layout = QHBoxLayout()
        self.threshold_label = QLabel("Blur Threshold:")
        self.threshold_input = QLineEdit("100")
        self.threshold_preset = QComboBox()
        self.threshold_preset.addItems(["Custom", "Low (50)", "Medium (100)", "High (150)"])
        self.threshold_preset.currentTextChanged.connect(self.update_threshold)
        threshold_layout.addWidget(self.threshold_label)
        threshold_layout.addWidget(self.threshold_input)
        threshold_layout.addWidget(self.threshold_preset)
        layout.addLayout(threshold_layout)

        self.progress_bar = QProgressBar()
        self.progress_bar.setVisible(False)
        layout.addWidget(self.progress_bar)

        self.start_button = QPushButton("Start Detection")
        self.start_button.clicked.connect(self.start_detection)
        layout.addWidget(self.start_button)

        self.image_label = QLabel()
        self.image_label.setAlignment(Qt.AlignCenter)
        self.image_label.setFixedSize(600, 400)
        layout.addWidget(self.image_label)

        self.image_info = QLabel()
        layout.addWidget(self.image_info)

        nav_layout = QHBoxLayout()
        self.prev_button = QPushButton("Previous")
        self.prev_button.clicked.connect(self.show_previous_image)
        self.keep_button = QPushButton("Keep")
        self.keep_button.clicked.connect(self.keep_image)
        self.reject_button = QPushButton("Move to Blurry")
        self.reject_button.clicked.connect(self.move_image)
        self.next_button = QPushButton("Next")
        self.next_button.clicked.connect(self.show_next_image)

        nav_layout.addWidget(self.prev_button)
        nav_layout.addWidget(self.keep_button)
        nav_layout.addWidget(self.reject_button)
        nav_layout.addWidget(self.next_button)
        layout.addLayout(nav_layout)

        self.log_output = QTextEdit()
        self.log_output.setReadOnly(True)
        self.log_output.setMaximumHeight(100)
        layout.addWidget(self.log_output)

        self.setLayout(layout)
        self.update_button_states(False)

    @staticmethod
    def load_image(image_path):
        try:
            if image_path.lower().endswith(('.cr2', '.nef', '.arw', '.dng', '.orf', '.raf')):
                with rawpy.imread(image_path) as raw:
                    rgb = raw.postprocess()
                    return cv2.cvtColor(rgb, cv2.COLOR_RGB2GRAY)
            return cv2.imread(image_path, cv2.IMREAD_GRAYSCALE)
        except Exception as e:
            print(f"Error loading {image_path}: {e}")
            return None

    def update_threshold(self, preset):
        if preset != "Custom":
            value = int(preset.split("(")[1].split(")")[0])
            self.threshold_input.setText(str(value))

    def update_button_states(self, enabled=True):
        for btn in [self.prev_button, self.keep_button,
                    self.reject_button, self.next_button]:
            btn.setEnabled(enabled)

    def start_detection(self):
        if not self.selected_folder:
            self.log_output.append("Please select a folder first!")
            return

        try:
            QApplication.processEvents()
            self.threshold_input.setText(self.threshold_input.text())
            threshold_text = self.threshold_input.text().strip()

            try:
                threshold = float(threshold_text)
                self.log_output.append(f"Using updated threshold: {threshold}")
            except ValueError:
                threshold = 100.0
                self.log_output.append("Invalid threshold! Using default (100).")

        except ValueError:
            self.log_output.append("Invalid threshold! Using default (100)")
            threshold = 100.0

        methods = []
        if self.laplacian_check.isChecked():
            methods.append('laplacian')
        if self.sobel_check.isChecked():
            methods.append('sobel')

        if not methods:
            self.log_output.append("Please select at least one detection method!")
            return

        self.progress_bar.setVisible(True)
        self.start_button.setEnabled(False)

        scan_subfolders = self.scan_subfolders_check.isChecked()
        self.log_output.append(f"{'Including' if scan_subfolders else 'Excluding'} subfolders in scan")

        self.analyzer_thread = ImageAnalyzerThread(
            self.selected_folder, threshold, methods, scan_subfolders)
        self.analyzer_thread.progress.connect(self.update_progress)
        self.analyzer_thread.finished.connect(self.detection_finished)
        self.analyzer_thread.start()

    def update_progress(self, value):
        self.progress_bar.setValue(value)

    def detection_finished(self, blurry_images):
        self.blurry_images = blurry_images
        self.progress_bar.setVisible(False)
        self.start_button.setEnabled(True)
        self.log_output.append(f"Found {len(self.blurry_images)} blurry images")

        if self.blurry_images:
            self.current_index = 0
            self.show_image()
            self.update_button_states(True)
        else:
            self.log_output.append("No blurry images found.")
            self.image_label.clear()  # Clear preview
            self.image_info.clear()  # Clear text info
            self.update_button_states(False)

    def show_image(self):
        if 0 <= self.current_index < len(self.blurry_images):
            img_path = self.blurry_images[self.current_index]
            img = Image.open(img_path)

            aspect_ratio = img.width / img.height
            if aspect_ratio > 1.5:
                new_width, new_height = 600, int(600 / aspect_ratio)
            else:
                new_width, new_height = int(400 * aspect_ratio), 400

            img = img.resize((new_width, new_height), Image.LANCZOS)
            pixmap = pil_to_qpixmap(img)
            self.image_label.setPixmap(pixmap)

            image_cv = cv2.imread(img_path, cv2.IMREAD_GRAYSCALE)
            sharpness_score = cv2.Laplacian(image_cv, cv2.CV_64F).var() if image_cv is not None else "N/A"

            file_size = os.path.getsize(img_path) / (1024 * 1024)
            self.image_info.setText(
                f"Image {self.current_index + 1}/{len(self.blurry_images)}\n"
                f"File: {os.path.basename(img_path)}\n"
                f"Path: {img_path}\n"
                f"Size: {file_size:.1f}MB | Dimensions: {img.width}x{img.height}\n"
                f"Sharpness Score: {sharpness_score:.2f} | Threshold: {self.threshold_input.text()}"
            )

            self.prev_button.setEnabled(self.current_index > 0)
            self.next_button.setEnabled(self.current_index < len(self.blurry_images) - 1)

    def browse_folder(self):
        folder = QFileDialog.getExistingDirectory(self, "Select Folder")
        if folder:
            self.selected_folder = folder
            self.folder_label.setText(f"Selected Folder: {folder}")

    def show_previous_image(self):
        if self.current_index > 0:
            self.current_index -= 1
            self.show_image()

    def show_next_image(self):
        if self.current_index < len(self.blurry_images) - 1:
            self.current_index += 1
            self.show_image()

    def keep_image(self):
        if 0 <= self.current_index < len(self.blurry_images):
            self.log_output.append(f"Kept: {os.path.basename(self.blurry_images[self.current_index])}")
            self.blurry_images.pop(self.current_index)
            if self.blurry_images:
                if self.current_index >= len(self.blurry_images):
                    self.current_index = len(self.blurry_images) - 1
                self.show_image()
            else:
                self.image_label.clear()
                self.image_info.clear()
                self.update_button_states(False)

    def move_image(self):
        if 0 <= self.current_index < len(self.blurry_images):
            src = self.blurry_images[self.current_index]

            src_folder = os.path.dirname(src)
            blurry_folder = os.path.join(src_folder, "Blurry_Images")
            os.makedirs(blurry_folder, exist_ok=True)

            dest_path = os.path.join(blurry_folder, os.path.basename(src))

            try:
                if os.path.exists(dest_path):
                    dest_path = self.get_unique_filename(dest_path = self.get_unique_filename(dest_path))

                shutil.move(str(src), str(dest_path))
                self.log_output.append(f"Moved to: {dest_path}")

                self.keep_image()
            except Exception as e:
                self.log_output.append(f"Error moving file: {e}")

    def get_unique_filename(self, path: str) -> str:
        base, ext = os.path.splitext(path)
        counter = 1

        # Keep generating new filenames until one is available
        while os.path.exists(f"{base}_{counter}{ext}"):
            counter += 1

        return f"{base}_{counter}{ext}"


if __name__ == "__main__":
    try:
        app = QApplication(sys.argv)
        window = BlurryImageDetector()
        window.show()
        sys.exit(app.exec_())
    except Exception as e:
        print("Critical Error:", e)
        import traceback
        traceback.print_exc()