import sys
import os
import json
import re
import base64
import socket  # Added for socket error handling
from cryptography.fernet import Fernet
from cryptography.hazmat.backends import default_backend
from cryptography.hazmat.primitives import hashes
from cryptography.hazmat.primitives.kdf.pbkdf2 import PBKDF2HMAC
from PyQt5.QtWidgets import (QApplication, QMainWindow, QWidget, QVBoxLayout,
                             QLineEdit, QTextEdit, QPushButton, QMessageBox, QLabel,
                             QFileDialog, QHBoxLayout, QTabWidget, QCheckBox, QScrollArea)
from PyQt5.QtCore import Qt
from PyQt5.QtGui import QFont
import smtplib
from email.mime.text import MIMEText
from email.mime.multipart import MIMEMultipart
from email.utils import formatdate, make_msgid


class HtmlEmailSenderApp(QMainWindow):
    def __init__(self):
        super().__init__()
        self.setWindowTitle("Αποστολή HTML Email")
        self.setGeometry(100, 100, 1000, 800)

        # Determine application path based on whether we're running from executable or script
        if getattr(sys, 'frozen', False):
            # Running as executable
            application_path = os.path.dirname(sys.executable)
        else:
            # Running as script
            application_path = os.path.dirname(os.path.abspath(__file__))

        # Config file paths in same directory as executable/script
        self.config_file = os.path.join(application_path, "email_settings.json")
        self.key_file = os.path.join(application_path, ".key")

        print(f"Config file path: {self.config_file}")
        print(f"Key file path: {self.key_file}")

        # Set up encryption key file
        self.setup_encryption()

        # Default SMTP settings with empty credentials
        self.smtp_settings = {
            'server': '',
            'port': 587,
            'username': '',
            'password': '',
            'from_name': '',
            'use_tls': True
        }

        # Load settings if they exist
        self.load_settings()

        # Create central widget and master layout
        central_widget = QWidget()
        self.setCentralWidget(central_widget)
        master_layout = QVBoxLayout(central_widget)

        # Create recipient email field - now using a text edit for multiple lines
        recipient_layout = QVBoxLayout()
        recipient_layout.addWidget(QLabel("Παραλήπτες (ένα email ανά γραμμή):"))
        self.recipient_emails = QTextEdit()
        self.recipient_emails.setPlaceholderText("Εισάγετε διευθύνσεις email, μία ανά γραμμή")
        self.recipient_emails.setMaximumHeight(100)  # Limit height
        recipient_layout.addWidget(self.recipient_emails)
        master_layout.addLayout(recipient_layout)

        # Create subject field
        subject_layout = QHBoxLayout()
        subject_layout.addWidget(QLabel("Θέμα:"))
        self.subject = QLineEdit()
        self.subject.setPlaceholderText("Θέμα")
        subject_layout.addWidget(self.subject)
        master_layout.addLayout(subject_layout)

        # Create tab widget for different views
        self.tab_widget = QTabWidget()

        # Create HTML editor tab
        html_widget = QWidget()
        html_layout = QVBoxLayout(html_widget)

        html_header_layout = QHBoxLayout()
        html_header_layout.addWidget(QLabel("Περιεχόμενο HTML:"))

        # Add load HTML file button
        self.load_button = QPushButton("Φόρτωση Αρχείου HTML")
        self.load_button.clicked.connect(self.load_html_file)
        html_header_layout.addWidget(self.load_button)

        html_layout.addLayout(html_header_layout)

        self.html_editor = QTextEdit()
        self.html_editor.setPlaceholderText(
            "Γράψτε το μήνυμά σας εδώ, επικολλήστε κώδικα HTML, ή φορτώστε ένα πρότυπο HTML - εσείς επιλέγετε!")
        self.html_editor.setMinimumHeight(500)
        html_layout.addWidget(self.html_editor)

        # Create a raw text editor tab
        text_widget = QWidget()
        text_layout = QVBoxLayout(text_widget)
        text_layout.addWidget(QLabel("Έκδοση Απλού Κειμένου:"))

        self.text_editor = QTextEdit()
        self.text_editor.setPlaceholderText(
            "Αυτή θα είναι η εναλλακτική έκδοση απλού κειμένου του email σας σε περίπτωση που ο email client του παραλήπτη δεν υποστηρίζει HTML...")
        self.text_editor.setMinimumHeight(500)
        text_layout.addWidget(self.text_editor)

        # Add auto-convert checkbox
        self.auto_convert_links = QCheckBox("Αυτόματη μετατροπή απλού κειμένου σε HTML")
        self.auto_convert_links.setChecked(True)
        text_layout.addWidget(self.auto_convert_links)

        # Connect signals after creating widgets
        self.text_editor.textChanged.connect(self.plain_text_changed)

        # Create server settings tab
        server_widget = QWidget()
        server_layout = QVBoxLayout(server_widget)
        server_layout.addWidget(QLabel("Ρυθμίσεις Διακομιστή Email:"))

        # SMTP server field
        smtp_layout = QHBoxLayout()
        smtp_layout.addWidget(QLabel("Διακομιστής SMTP:"))
        self.smtp_server = QLineEdit(self.smtp_settings['server'])
        smtp_layout.addWidget(self.smtp_server)
        server_layout.addLayout(smtp_layout)

        # Port field
        port_layout = QHBoxLayout()
        port_layout.addWidget(QLabel("Θύρα SMTP:"))
        self.smtp_port = QLineEdit(str(self.smtp_settings['port']))
        port_layout.addWidget(self.smtp_port)
        server_layout.addLayout(port_layout)

        # Username field
        username_layout = QHBoxLayout()
        username_layout.addWidget(QLabel("Όνομα Χρήστη Email:"))
        self.smtp_username = QLineEdit(self.smtp_settings['username'])
        username_layout.addWidget(self.smtp_username)
        server_layout.addLayout(username_layout)

        # Password field
        password_layout = QHBoxLayout()
        password_layout.addWidget(QLabel("Κωδικός Πρόσβασης Email:"))
        self.smtp_password = QLineEdit(self.smtp_settings['password'])
        self.smtp_password.setEchoMode(QLineEdit.Password)
        password_layout.addWidget(self.smtp_password)
        server_layout.addLayout(password_layout)

        # From name field
        from_name_layout = QHBoxLayout()
        from_name_layout.addWidget(QLabel("Όνομα Αποστολέα:"))
        self.from_name = QLineEdit(self.smtp_settings.get('from_name', ''))
        from_name_layout.addWidget(self.from_name)
        server_layout.addLayout(from_name_layout)

        # SSL/TLS options
        security_layout = QHBoxLayout()
        security_layout.addWidget(QLabel("Ασφάλεια:"))
        self.use_tls = QCheckBox("Χρήση TLS")
        self.use_tls.setChecked(self.smtp_settings.get('use_tls', True))
        security_layout.addWidget(self.use_tls)
        server_layout.addLayout(security_layout)

        # Server button layout
        server_buttons_layout = QHBoxLayout()

        # Save button
        save_button = QPushButton("Αποθήκευση Ρυθμίσεων")
        save_button.clicked.connect(self.save_smtp_settings)
        server_buttons_layout.addWidget(save_button)

        # Test connection button
        test_connection_button = QPushButton("Δοκιμή Σύνδεσης")
        test_connection_button.clicked.connect(self.test_smtp_connection)
        server_buttons_layout.addWidget(test_connection_button)

        # Test email button
        test_email_button = QPushButton("Αποστολή Δοκιμαστικού Email")
        test_email_button.clicked.connect(self.send_test_email)
        server_buttons_layout.addWidget(test_email_button)

        server_layout.addLayout(server_buttons_layout)

        # Add spacer at the bottom
        server_layout.addStretch()

        # Create GDPR tab
        gdpr_widget = QWidget()
        gdpr_layout = QVBoxLayout(gdpr_widget)

        # Create a scroll area for the GDPR content
        gdpr_scroll = QScrollArea()
        gdpr_scroll.setWidgetResizable(True)
        gdpr_scroll.setFrameShape(QScrollArea.NoFrame)

        gdpr_content = QWidget()
        gdpr_content_layout = QVBoxLayout(gdpr_content)

        # GDPR Title
        gdpr_title = QLabel("Πληροφορίες για το GDPR και την Αποστολή Email")
        gdpr_title.setStyleSheet("font-size: 16pt; font-weight: bold; margin-bottom: 10px;")
        gdpr_content_layout.addWidget(gdpr_title)

        # GDPR Introduction
        gdpr_intro = QLabel("""Ο Γενικός Κανονισμός Προστασίας Δεδομένων (GDPR) είναι ένας κανονισμός της Ευρωπαϊκής 
Ένωσης σχετικά με την προστασία των δεδομένων και την ιδιωτικότητα στην ΕΕ και τον ΕΟΧ. 
Αφορά επίσης τη μεταφορά προσωπικών δεδομένων εκτός των περιοχών ΕΕ/ΕΟΧ.""")
        gdpr_intro.setWordWrap(True)
        gdpr_content_layout.addWidget(gdpr_intro)

        # Email Marketing and GDPR
        gdpr_email_title = QLabel("Email Marketing και GDPR")
        gdpr_email_title.setStyleSheet("font-size: 14pt; font-weight: bold; margin-top: 20px; margin-bottom: 10px;")
        gdpr_content_layout.addWidget(gdpr_email_title)

        gdpr_email_content = QLabel("""Κατά την αποστολή email μάρκετινγκ, πρέπει να συμμορφώνεστε με το GDPR. 
Αυτό περιλαμβάνει:

1. Λήψη συγκατάθεσης: Πρέπει να έχετε ρητή συγκατάθεση από τους παραλήπτες για να τους στείλετε email μάρκετινγκ.

2. Δικαίωμα απόσυρσης συγκατάθεσης: Κάθε email πρέπει να περιλαμβάνει έναν εύκολο τρόπο για τους παραλήπτες να 
απεγγραφούν από τη λίστα σας.

3. Διαφάνεια: Πρέπει να είναι σαφές ποιος είναι ο αποστολέας του email και για ποιο σκοπό αποστέλλεται.

4. Τήρηση αρχείων: Πρέπει να διατηρείτε αρχεία που αποδεικνύουν τη συγκατάθεση.

5. Ασφάλεια δεδομένων: Τα προσωπικά δεδομένα των παραληπτών πρέπει να αποθηκεύονται με ασφάλεια.""")
        gdpr_email_content.setWordWrap(True)
        gdpr_content_layout.addWidget(gdpr_email_content)

        # Best Practices
        gdpr_practices_title = QLabel("Βέλτιστες Πρακτικές για Συμμόρφωση με το GDPR")
        gdpr_practices_title.setStyleSheet("font-size: 14pt; font-weight: bold; margin-top: 20px; margin-bottom: 10px;")
        gdpr_content_layout.addWidget(gdpr_practices_title)

        gdpr_practices_content = QLabel("""Για να διασφαλίσετε τη συμμόρφωση με το GDPR κατά τη χρήση αυτής της εφαρμογής:

1. Χρησιμοποιήστε μόνο λίστες email όπου οι παραλήπτες έχουν δώσει ρητή συγκατάθεση για λήψη emails.

2. Συμπεριλάβετε πληροφορίες για την ταυτότητά σας στο email.

3. Συμπεριλάβετε έναν σύνδεσμο απεγγραφής σε κάθε email που στέλνετε.

4. Συμπεριλάβετε τη φυσική σας διεύθυνση στο email.

5. Διατηρήστε αρχείο συγκατάθεσης για κάθε παραλήπτη.

6. Διαγράψτε τις διευθύνσεις email όσων έχουν απεγγραφεί ή αποσύρει τη συγκατάθεσή τους.

7. Σε περίπτωση παραβίασης δεδομένων, ενημερώστε άμεσα τις αρμόδιες αρχές και τους επηρεαζόμενους παραλήπτες.""")
        gdpr_practices_content.setWordWrap(True)
        gdpr_content_layout.addWidget(gdpr_practices_content)

        # Legal Disclaimer
        gdpr_disclaimer_title = QLabel("Νομική Αποποίηση Ευθυνών")
        gdpr_disclaimer_title.setStyleSheet(
            "font-size: 14pt; font-weight: bold; margin-top: 20px; margin-bottom: 10px;")
        gdpr_content_layout.addWidget(gdpr_disclaimer_title)

        gdpr_disclaimer_content = QLabel("""Αυτή η εφαρμογή παρέχεται "ως έχει", χωρίς εγγυήσεις οποιουδήποτε είδους. 
Η χρήση αυτής της εφαρμογής δεν αποτελεί νομική συμβουλή. Η συμμόρφωση με το GDPR και όλους τους ισχύοντες νόμους και 
κανονισμούς προστασίας δεδομένων αποτελεί αποκλειστική ευθύνη του χρήστη. Συνιστούμε να συμβουλευτείτε έναν νομικό 
σύμβουλο για συγκεκριμένες συμβουλές σχετικά με τη συμμόρφωση με το GDPR.""")
        gdpr_disclaimer_content.setWordWrap(True)
        gdpr_content_layout.addWidget(gdpr_disclaimer_content)

        # Resources
        gdpr_resources_title = QLabel("Χρήσιμες Πηγές")
        gdpr_resources_title.setStyleSheet("font-size: 14pt; font-weight: bold; margin-top: 20px; margin-bottom: 10px;")
        gdpr_content_layout.addWidget(gdpr_resources_title)

        gdpr_resources_content = QLabel("""Για περισσότερες πληροφορίες σχετικά με το GDPR και την αποστολή email:

• Επίσημη ιστοσελίδα GDPR της ΕΕ: https://gdpr.eu/
• Ελληνική Αρχή Προστασίας Δεδομένων Προσωπικού Χαρακτήρα: https://www.dpa.gr/
• Οδηγός GDPR για μικρές επιχειρήσεις: https://ec.europa.eu/info/law/law-topic/data-protection/reform/rules-business-and-organisations_el""")
        gdpr_resources_content.setWordWrap(True)
        gdpr_content_layout.addWidget(gdpr_resources_content)

        # Add space at the bottom
        gdpr_content_layout.addStretch()

        # Set the content widget to the scroll area
        gdpr_scroll.setWidget(gdpr_content)
        gdpr_layout.addWidget(gdpr_scroll)

        # Add the tabs
        self.tab_widget.addTab(html_widget, "Περιεχόμενο HTML")
        self.tab_widget.addTab(text_widget, "Έκδοση Απλού Κειμένου")
        self.tab_widget.addTab(server_widget, "Ρυθμίσεις Διακομιστή")
        self.tab_widget.addTab(gdpr_widget, "GDPR & Συμμόρφωση")

        master_layout.addWidget(self.tab_widget)

        # Add action buttons
        buttons_layout = QHBoxLayout()

        # Create send button
        self.send_button = QPushButton("Αποστολή Email")
        self.send_button.clicked.connect(self.send_email)
        buttons_layout.addWidget(self.send_button)

        master_layout.addLayout(buttons_layout)

        # Add credit label at bottom right
        credit_layout = QHBoxLayout()
        credit_layout.addStretch()  # Push label to the right

        credit_label = QLabel("Created by @BinaryMind")
        # Style the label
        credit_label.setStyleSheet("color: #888888;")  # Light gray color
        font = credit_label.font()
        font.setItalic(True)
        font.setPointSize(8)
        credit_label.setFont(font)

        credit_layout.addWidget(credit_label)
        master_layout.addLayout(credit_layout)

    def setup_encryption(self):
        """Setup encryption key or load existing one"""
        try:
            # Check if key file exists, create if it doesn't
            if not os.path.exists(self.key_file):
                print(f"Creating new encryption key at {self.key_file}")
                # Generate a new key
                key = Fernet.generate_key()
                with open(self.key_file, 'wb') as f:
                    f.write(key)
                # Set permissions to limit access (on Unix systems)
                try:
                    os.chmod(self.key_file, 0o600)  # Only owner can read/write
                except:
                    pass  # Skip if on Windows or permission change fails
            else:
                print(f"Using existing encryption key at {self.key_file}")

            # Load the key
            with open(self.key_file, 'rb') as f:
                self.key = f.read()

            # Create Fernet cipher
            self.cipher = Fernet(self.key)
            print("Encryption setup successful")

        except Exception as e:
            print(f"Σφάλμα ρύθμισης κρυπτογράφησης: {e}")
            # If encryption setup fails, fall back to no encryption
            self.cipher = None
            print("Falling back to no encryption")

    def encrypt_password(self, password):
        """Encrypt the password string"""
        if not password or not self.cipher:
            return password

        try:
            return self.cipher.encrypt(password.encode()).decode()
        except Exception as e:
            print(f"Σφάλμα κρυπτογράφησης: {e}")
            return password

    def decrypt_password(self, encrypted_password):
        """Decrypt the password string"""
        if not encrypted_password or not self.cipher:
            return encrypted_password

        try:
            # Check if it's encrypted (starts with 'gAAAAA')
            if isinstance(encrypted_password, str) and encrypted_password.startswith('gAAAAA'):
                return self.cipher.decrypt(encrypted_password.encode()).decode()
            return encrypted_password
        except Exception as e:
            print(f"Σφάλμα αποκρυπτογράφησης: {e}")
            return encrypted_password

    def load_settings(self):
        """Load SMTP settings from the configuration file"""
        try:
            if os.path.exists(self.config_file):
                print(f"Loading settings from {self.config_file}")
                with open(self.config_file, 'r') as f:
                    saved_settings = json.load(f)

                    # Update settings, keeping defaults for any missing keys
                    for key, value in saved_settings.items():
                        if key == 'password' and value:
                            # Decrypt the password
                            self.smtp_settings[key] = self.decrypt_password(value)
                        else:
                            self.smtp_settings[key] = value

                print(f"Φορτώθηκαν ρυθμίσεις από {self.config_file}")
            else:
                print(f"No settings file found at {self.config_file}, using defaults")
        except Exception as e:
            print(f"Σφάλμα φόρτωσης ρυθμίσεων: {e}")

    def save_settings(self):
        """Save SMTP settings to the configuration file"""
        try:
            # Update SMTP settings from UI fields
            self.smtp_settings['server'] = self.smtp_server.text()
            try:
                self.smtp_settings['port'] = int(self.smtp_port.text())
            except ValueError:
                self.smtp_settings['port'] = 587  # Default to 587 if invalid
            self.smtp_settings['username'] = self.smtp_username.text()
            # Encrypt the password before saving
            self.smtp_settings['password'] = self.encrypt_password(self.smtp_password.text())
            self.smtp_settings['from_name'] = self.from_name.text()
            self.smtp_settings['use_tls'] = self.use_tls.isChecked()

            print(f"Saving settings to {self.config_file}")

            # Write to file
            with open(self.config_file, 'w') as f:
                json.dump(self.smtp_settings, f, indent=4)

            QMessageBox.information(self, "Επιτυχία", "Οι ρυθμίσεις SMTP αποθηκεύτηκαν με επιτυχία!")
            print("Settings saved successfully")
        except Exception as e:
            QMessageBox.critical(self, "Σφάλμα", f"Αποτυχία αποθήκευσης ρυθμίσεων: {str(e)}")
            print(f"Error saving settings: {e}")

    def save_smtp_settings(self):
        """Save SMTP settings and update the current session"""
        self.save_settings()

    def plain_text_changed(self):
        """Convert plain text to HTML when auto-convert is enabled"""
        if self.auto_convert_links.isChecked():
            plain_text = self.text_editor.toPlainText()
            if plain_text:
                html = self.convert_text_to_html(plain_text)
                self.html_editor.setText(html)

    def convert_text_to_html(self, text):
        """Convert plain text to HTML with clickable links"""
        # Escape HTML special characters
        html_text = text.replace('&', '&amp;')
        html_text = html_text.replace('<', '&lt;')
        html_text = html_text.replace('>', '&gt;')

        # Convert newlines to <br> tags
        html_text = html_text.replace('\n', '<br>\n')

        # URL pattern for detecting links
        url_pattern = r'(https?://[^\s]+|www\.[^\s]+\.[^\s]+)'

        # Replace URLs with HTML links
        def replace_url(match):
            url = match.group(1)
            if url.startswith('www.'):
                link_url = 'http://' + url
            else:
                link_url = url
            return f'<a href="{link_url}" style="color:#0066cc; text-decoration:underline;">{url}</a>'

        html_text = re.sub(url_pattern, replace_url, html_text)

        # Wrap in basic HTML structure
        html = f"""<!DOCTYPE html>
<html>
<head>
    <meta charset="utf-8">
    <style>
        body {{
            font-family: Arial, Helvetica, sans-serif;
            font-size: 14px;
            line-height: 1.5;
            color: #333333;
        }}
        a {{
            color: #0066cc;
            text-decoration: underline;
        }}
    </style>
</head>
<body>
    {html_text}
</body>
</html>"""

        return html

    def load_html_file(self):
        file_path, _ = QFileDialog.getOpenFileName(
            self, "Άνοιγμα Αρχείου HTML", "", "Αρχεία HTML (*.html);;Όλα τα Αρχεία (*)"
        )

        if file_path:
            try:
                with open(file_path, 'r', encoding='utf-8') as file:
                    html_content = file.read()
                    self.html_editor.setText(html_content)

                    # Generate a simple plain text version by stripping HTML tags
                    # Basic HTML tag removal (not perfect but works for simple HTML)
                    plain_text = re.sub(r'<.*?>', '', html_content)
                    # Replace HTML entities
                    plain_text = plain_text.replace('&nbsp;', ' ')
                    plain_text = plain_text.replace('&lt;', '<')
                    plain_text = plain_text.replace('&gt;', '>')
                    plain_text = plain_text.replace('&amp;', '&')
                    # Remove excess whitespace
                    plain_text = re.sub(r'\s+', ' ', plain_text)

                    self.text_editor.setText(plain_text)

                    QMessageBox.information(self, "Επιτυχία", f"Φορτώθηκε HTML από {file_path}")
            except Exception as e:
                QMessageBox.critical(self, "Σφάλμα", f"Αποτυχία φόρτωσης αρχείου: {str(e)}")

    def test_smtp_connection(self):
        try:
            # Get current settings from UI fields (don't save them yet)
            server_address = self.smtp_server.text()
            port = int(self.smtp_port.text())
            username = self.smtp_username.text()
            password = self.smtp_password.text()  # Get the raw password from the UI
            use_tls = self.use_tls.isChecked()

            # Create SMTP session
            server = smtplib.SMTP(server_address, port)

            # Start TLS if selected
            if use_tls:
                server.starttls()

            # Login to the server
            server.login(username, password)

            # Quit the server
            server.quit()

            QMessageBox.information(
                self,
                "Επιτυχής Σύνδεση",
                f"Επιτυχής σύνδεση στο {server_address}:{port} με όνομα χρήστη {username}"
            )

        except Exception as e:
            QMessageBox.critical(self, "Αποτυχία Σύνδεσης", f"Αποτυχία σύνδεσης: {str(e)}")

    def send_test_email(self):
        """Send a test email to the user's own email address"""
        test_email = self.smtp_username.text()  # Send to self as test
        if not test_email:
            QMessageBox.warning(self, "Προειδοποίηση", "Παρακαλώ εισάγετε το όνομα χρήστη email σας παραπάνω.")
            return

        # Simple test content
        subject = "Δοκιμαστικό Email από την Εφαρμογή Αποστολής HTML Email"
        plain_text = "Αυτό είναι ένα δοκιμαστικό email από την εφαρμογή σας Αποστολής HTML Email. Αν το βλέπετε, η ρύθμιση email σας λειτουργεί σωστά!"
        html_content = f"""
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset="utf-8">
            <style>
                body {{
                    font-family: Arial, Helvetica, sans-serif;
                    font-size: 14px;
                    line-height: 1.5;
                    color: #333333;
                }}
                h1 {{
                    color: #0066cc;
                }}
                .success {{
        color: green;
                    font-weight: bold;
                }}
            </style>
        </head>
        <body>
            <h1>Δοκιμαστικό Email</h1>
            <p>Αυτό είναι ένα δοκιμαστικό email από την εφαρμογή σας Αποστολής HTML Email.</p>
            <p class="success">Αν το βλέπετε, η ρύθμιση email σας λειτουργεί σωστά!</p>
            <p>Διακομιστής: {self.smtp_server.text()}<br>
            Όνομα Χρήστη: {test_email}</p>
        </body>
        </html>
        """

        try:
            # Get settings from UI
            server_address = self.smtp_server.text()
            port = int(self.smtp_port.text())
            username = self.smtp_username.text()
            password = self.smtp_password.text()
            use_tls = self.use_tls.isChecked()
            from_name = self.from_name.text() or "Αποστολή Email"

            # Create SMTP session
            server = smtplib.SMTP(server_address, port)

            # Start TLS if selected
            if use_tls:
                server.starttls()

            # Login to the server
            server.login(username, password)

            # Create a new message
            msg = MIMEMultipart('alternative')

            # Get domain for Message-ID from username
            email_domain = username.split('@')[-1]

            # Add headers
            msg['From'] = f"{from_name} <{username}>"
            msg['To'] = test_email
            msg['Subject'] = subject
            msg['Date'] = formatdate(localtime=True)
            msg['Message-ID'] = make_msgid(domain=email_domain)

            # Attach parts - plain first, then HTML
            msg.attach(MIMEText(plain_text, 'plain', 'utf-8'))
            msg.attach(MIMEText(html_content, 'html', 'utf-8'))

            # Send email
            text = msg.as_string()
            server.sendmail(username, test_email, text)
            server.quit()

            QMessageBox.information(
                self,
                "Το Δοκιμαστικό Email Εστάλη",
                f"Ένα δοκιμαστικό email εστάλη με επιτυχία στο {test_email}.\n\nΠαρακαλώ ελέγξτε τα εισερχόμενά σας για να επιβεβαιώσετε ότι το λάβατε."
            )

        except Exception as e:
            QMessageBox.critical(self, "Σφάλμα", f"Αποτυχία αποστολής δοκιμαστικού email: {str(e)}")

    def send_email(self):
        # Get recipients from text edit - one email per line
        recipient_text = self.recipient_emails.toPlainText().strip()
        if not recipient_text:
            QMessageBox.warning(self, "Προειδοποίηση", "Παρακαλώ εισάγετε τουλάχιστον μία διεύθυνση email παραλήπτη.")
            return

        # Split by newlines and filter out empty lines
        recipients = [email.strip() for email in recipient_text.split('\n') if email.strip()]

        subject = self.subject.text()
        if not subject:
            QMessageBox.warning(self, "Προειδοποίηση", "Παρακαλώ εισάγετε ένα θέμα email.")
            return

        self.send_html_email(recipients, subject)

    def send_html_email(self, recipients, subject):
        try:
            html_content = self.html_editor.toPlainText()
            plain_text = self.text_editor.toPlainText()

            if not html_content:
                QMessageBox.warning(self, "Προειδοποίηση", "Το περιεχόμενο HTML είναι κενό.")
                return

            # Create a progress message box but don't show it yet
            progress_msg = QMessageBox(self)
            progress_msg.setIcon(QMessageBox.Information)
            progress_msg.setWindowTitle("Αποστολή Email")
            progress_msg.setText(f"Αποστολή σε {len(recipients)} παραλήπτες...")

            # Only show progress dialog for multiple recipients
            if len(recipients) > 1:
                progress_msg.setStandardButtons(QMessageBox.NoButton)
                progress_msg.show()
                QApplication.processEvents()  # Update the UI

            # Always get the most current values from the UI
            server_address = self.smtp_server.text()
            port = int(self.smtp_port.text())
            username = self.smtp_username.text()
            password = self.smtp_password.text()  # Get directly from the UI, not from saved settings
            use_tls = self.use_tls.isChecked()
            from_name = self.from_name.text()

            # Create SMTP session
            server = smtplib.SMTP(server_address, port)

            # Start TLS if enabled
            if use_tls:
                server.starttls()

            # Login to the server
            server.login(username, password)

            # Track successful and failed emails
            successful = []
            failed = []

            # Send to each recipient individually
            for recipient in recipients:
                try:
                    # Create a new message for each recipient
                    msg = MIMEMultipart('alternative')

                    # Get domain for Message-ID from username
                    email_domain = username.split('@')[-1]

                    # Add headers
                    msg['From'] = f"{from_name} <{username}>"
                    msg['To'] = recipient
                    msg['Subject'] = subject
                    msg['Date'] = formatdate(localtime=True)
                    msg['Message-ID'] = make_msgid(domain=email_domain)

                    # Attach parts - plain first, then HTML
                    msg.attach(MIMEText(plain_text, 'plain', 'utf-8'))
                    msg.attach(MIMEText(html_content, 'html', 'utf-8'))

                    # Send email
                    text = msg.as_string()
                    server.sendmail(username, recipient, text)
                    successful.append(recipient)

                except Exception as e:
                    failed.append(f"{recipient} (Σφάλμα: {str(e)})")

                # Update UI periodically
                if len(recipients) > 1 and progress_msg.isVisible():
                    progress_msg.setText(
                        f"Αποστολή σε {len(recipients)} παραλήπτες...\n\nΟλοκληρώθηκαν: {len(successful) + len(failed)}/{len(recipients)}")
                    QApplication.processEvents()

            # Close the SMTP connection
            server.quit()

            # Update progress message with result and add OK button
            if len(recipients) > 1 and progress_msg.isVisible():
                if failed:
                    result_text = f"Ολοκληρώθηκε: {len(successful)}/{len(recipients)} επιτυχείς, {len(failed)}/{len(recipients)} αποτυχίες."
                    progress_msg.setIcon(QMessageBox.Warning)
                    progress_msg.setWindowTitle("Η Αποστολή Ολοκληρώθηκε - Με Σφάλματα")
                else:
                    result_text = f"Επιτυχής αποστολή σε όλους τους {len(recipients)} παραλήπτες!"
                    progress_msg.setWindowTitle("Η Αποστολή Ολοκληρώθηκε")

                progress_msg.setText(result_text)
                progress_msg.setStandardButtons(QMessageBox.Ok)
                progress_msg.exec_()  # This will block until user clicks OK
            else:
                # If progress dialog wasn't shown, use a regular message box
                if failed:
                    error_msg = "Ορισμένα email απέτυχαν να αποσταλούν:\n\n" + "\n".join(failed)
                    if successful:
                        error_msg += f"\n\nΕπιτυχής αποστολή σε {len(successful)} παραλήπτη(ες)."
                    QMessageBox.warning(self, "Μερική Επιτυχία", error_msg)
                else:
                    QMessageBox.information(
                        self,
                        "Επιτυχία",
                        f"Το email εστάλη με επιτυχία σε {len(successful)} παραλήπτη(ες)!"
                    )

        except Exception as e:
            QMessageBox.critical(self, "Σφάλμα", f"Αποτυχία αποστολής email: {str(e)}")


if __name__ == '__main__':
    app = QApplication(sys.argv)
    window = HtmlEmailSenderApp()
    window.show()
    sys.exit(app.exec_())